<?php

namespace App\Http\Controllers;

use App\Models\File;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class FileController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): JsonResponse
    {
        $files = File::all()->load(['products', 'stocks', 'categories']);

        return response()->json(
            [
                'success' => true,
                'data' => $files,
            ]
        );
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        $rules = [
            'file' => 'required|file|mimes:webp,png,jpg,jpeg,pdf,doc,docx,xls,xlsx,ppt,pptx',
        ];

        $validator = \Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json(
                [
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ],
                400
            );
        }

        $file = $request->file('file');

        $path = $file->store('files');

        $data = [
            'path' => $path,
            'name' => $file->getClientOriginalName(),
            'mime_type' => $file->getMimeType(),
        ];

        $file = File::create($data);
        return response()->json([
            'success' => true,
            'data' => $file,
        ], 201);

    }

    public function multiple(Request $request): JsonResponse
    {
        // \Log::info($request->all());

        $rules = [
            'files' => 'required',
            'files.*' => 'mimes:doc,docx,pdf,jpg,jpeg,png,webp',
        ];

        $validator = \Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json(
                [
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ],
                400
            );
        }

        $files = $request->file('files');

        \Log::info($files);

        $filesUploaded = [];

        foreach ($files as $file) {
            $path = $file->store('files');
            $data = [
                'path' => $path,
                // 'name' => $file->getClientOriginalName(),
                'name' => $file->hashName(),
                'mime_type' => $file->getMimeType(),
            ];

            $fileToSave = new File($data);
            $fileToSave->save();

            array_push($filesUploaded, $fileToSave);
        }



        return response()->json([
            'success' => true,
            'data' => $filesUploaded,
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(File $file): JsonResponse
    {
        //
        return response()->json([
            'success' => true,
            'data' => $file,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, File $file): JsonResponse
    {
        $rules = [
            'file' => 'required|file|mimes:png,jpg,jpeg,pdf,doc,docx,xls,xlsx,ppt,pptx',
        ];

        $validator = \Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json(
                [
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ],
                400
            );
        }

        $fileUploaded = $request->file('file');

        $path = $file->store('files');

        $data = [
            'path' => $path,
            'name' => $fileUploaded->getClientOriginalName(),
            'mime_type' => $fileUploaded->getMimeType(),
        ];

        \Storage::delete($file->path);

        $file->update($data);


        return response()->json([
            'success' => true,
            'data' => $file,
        ], 201);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(File $file): JsonResponse
    {
        $file->delete();

        \Storage::delete($file->path);

        return response()->json([
            'success' => true,
            'message' => 'File deleted successfully',
        ]);
    }
}
