<?php

namespace App\Http\Controllers\Sales;

use App\Http\Controllers\Controller;
use App\Http\Requests\CreateSaleRequest;
use App\Models\Coupon;
use App\Models\Customer;
use App\Models\Sale;
use Dotenv\Exception\ValidationException;
use Illuminate\Http\JsonResponse;


class CreateSaleController extends Controller
{


    public function createSale(CreateSaleRequest $request): JsonResponse
    {
        \DB::beginTransaction();
        try {

            $banks = $request->banks ?? [];

            $banks_total = isset($banks) ? array_sum(array_column($banks, 'amount')) : 0;

            $total = $request->total;

            if ($request->use_balance ?? false) {
                $total = $this->checkBalance($request);
            }

            if ($banks_total > $total) {
                throw new ValidationException('El monto total de los bancos no coincide con el monto total de la venta');
            }

            $isPaid = $total - $banks_total == 0 ? true : false;
            $isPaid = $request->payment_type === 'Pago en efectivo' ? true : $isPaid;
            $isPaid = isset($request->sale_category) &&
                $request->sale_category != 'purchase' ? false : $isPaid;


            if (isset($request->coupon_id)) {
                $couponDiscount = $this->checkCupon($request);
                $total -= $couponDiscount;
            }

            if ($total < 0) {
                throw new \Exception("El monto total no puede ser menor a 0");
            }


            $sale = Sale::create([
                'total' => $request->total,
                'sale_type' => $request->sale_type,
                'customer_id' => $request->customer_id,
                'user_id' => $request->user_id ?? null,
                'pricetype_id' => $request->pricetype_id ?? null,
                'payment_type' => $request->payment_type ?? null,
                'bank_id' => $request->bank_id ?? null,
                'sale_category' => $request->sale_category ?? null,
                'shipping_total' => $request->shipping_total,
                'coupon_id' => $request->coupon_id ?? null,
                'status' => $isPaid ? 'paid' : 'pending',
                'use_balance' => $request->use_balance ?? false,
                'balance_used' => $request->use_balance ? $request->total - $total : 0,
                'pending_applied_to_balance' => $request->apply_pending_to_balance ?? false,
            ]);



            if (isset($request->sale_details)) {
                $this->createSaleDetails($request, $sale);
            }

            if (
                isset($request->banks) &&
                $sale->sale_category != 'quotation'
            ) {
                $this->attachBanks($request, $sale);
            }

            if (
                $request->apply_pending_to_balance &&
                $banks_total < $total
            ) {
                $this->discountPendigSaleAmountFromBalance(
                    $sale,
                    $total - $banks_total
                );
            }

            \DB::commit();

            $sale->load('sale_details');

            return response()->json([
                'success' => true,
                'message' => 'Venta creada con éxito',
                'request' => $request->all(),
                'total' => $total,
                'data' => $sale,
            ], 201);
        } catch (\Throwable $th) {
            \DB::rollBack();
            \Log::error($th);
            return response()->json([
                'success' => false,
                'message' => 'Error al crear la venta',
                'error' => $th->getMessage()
            ], 400);
        }

    }

    private function checkBalance(CreateSaleRequest $request): float
    {

        $customer = Customer::find($request->customer_id);
        $prev_balance = $customer->balance;

        // discount balance from total
        if ($request->total > $customer->balance) {
            $customer->update([
                'balance' => 0,
            ]);

            return $request->total - $prev_balance;
        }

        $customer->update([
            'balance' => $customer->balance - $request->total,
        ]);

        return 0;
    }

    private function discountPendigSaleAmountFromBalance(
        Sale $sale,
        float $pendingAmountToBalance
    ): void {
        $sale->update([
            'status' => 'pending',
        ]);

        $customer = Customer::find($sale->customer_id);

        $customer->update([
            'balance' => $customer->balance - $pendingAmountToBalance,
        ]);
    }

    private function checkCupon(CreateSaleRequest $request): float
    {


        $coupon_query = Coupon::query();

        $coupon_query->where('id', $request->coupon_id);
        $coupon_query->where('status', 'active');

        $coupon = $coupon_query->first();

        \Log::info($coupon);

        if (!$coupon) {
            throw new \Exception("Cupón no válido o ya alcanzó su límite de uso");
        }

        $coupon->update([
            'used' => $coupon->used + 1,
        ]);

        if ($coupon->used == $coupon->use_limit) {
            $coupon->update([
                'status' => 'inactive',
            ]);
        }

        if ($coupon->expires_at < now()) {
            throw new \Exception("El cupón ya expiró");
        }

        if ($coupon->used > $coupon->use_limit) {
            throw new \Exception("El cupón ya alcanzó su límite de uso");
        }

        if ($coupon->type == 'percent') {
            return $request->total * ($coupon->value / 100);
        } else {
            return $coupon->value;
        }


    }

    private function createSaleDetails(CreateSaleRequest $request, Sale $sale)
    {

        foreach ($request->sale_details as $sale_details) {
            $sale_detail = $sale->sale_details()->create([
                'product_id' => $sale_details['product']['id'],
                // $sale_details['product_id'],
                'quantity' => $sale_details['quantity'],
                'total' => $sale_details['price'],
                'cart_item' => json_encode($sale_details['product']),
                'stock_id' => $sale_details['stock_id'],
                'size_id' => $sale_details['size_id'],

            ]);

            // if is quotation not update stock
            if ($sale->sale_category == 'quotation') {
                continue;
            }

            $sale_detail->stock->update([
                'stock' => $sale_detail->stock->stock - $sale_detail->quantity,
            ]);

            // if sale stock has 0 stock, throw error
            if ($sale_detail->stock->stock < 0) {
                throw new \Exception("No hay stock suficiente para el producto {$sale_detail->product->name} en color {$sale_detail->stock->color->name}");
            }
        }
    }

    private function attachBanks(CreateSaleRequest $request, Sale $sale)
    {
        foreach ($request->banks as $bank) {
            $sale->banks()->attach($bank['bank_id'], ['amount' => $bank['amount']]);
        }
    }

}