<?php

namespace App\Http\Controllers\Sales;

use App\Http\Controllers\Controller;
use App\Models\Coupon;
use App\Models\Sale;
use App\Models\Sale_detail;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class SaleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $rules = [
            "start_date" => "nullable|date",
            "end_date" => "nullable|date",
            "customer_id" => "nullable|exists:customers,id",
            "user_id" => "nullable|exists:users,id",
            "payment_type" => "nullable|string",
            "bank_id" => "nullable|exists:banks,id",
            "supplier_name" => "nullable|string",
            "sale_category" => "nullable|string:purchase,quotation,layaway",
            "status" => "nullable|string",
        ];


        $validator = \Validator::make(request()->query(), $rules);

        $data = $validator->validated();

        $sales_query = Sale::query();

        if (isset($data['start_date']) && isset($data['end_date'])) {
            $sales_query->whereBetween('created_at', [$data['start_date'], $data['end_date']]);
        }

        if (isset($data['customer_id'])) {
            $sales_query->where('customer_id', $data['customer_id']);
        }

        if (isset($data['user_id'])) {
            $sales_query->where('user_id', $data['user_id']);
        }

        if (isset($data['status'])) {
            $sales_query->where('status', $data['status']);
        }

        if (isset($data['payment_type'])) {
            $sales_query->where('payment_type', $data['payment_type']);
        } else {
            $sales_query->where('payment_type', '<>', 'Pago a crédito');
        }

        if (isset($data['sale_category'])) {
            $sales_query->where('sale_category', $data['sale_category']);
        } else {
            $sales_query->whereNotIn('sale_category', ['quotation', 'layaway']);
        }

        if (isset($data['bank_id'])) {
            $sales_query->whereHas('banks', function ($query) use ($data) {
                $query->where('bank_id', $data['bank_id']);
            });
        }

        if (isset($data['supplier_name'])) {
            $sales_query->whereHas('sale_details', function ($query) use ($data) {
                $query->whereHas('product', function ($query) use ($data) {
                    $query->where("supplier_name", $data['supplier_name']);
                });
            });
        }

        $sales = $sales_query->orderBy('id', 'desc')->get()->load([
            'sale_details',
            'sale_details.product',
            'sale_details.stock.color',
            'sale_details.stock.size',
            'customer',
            'user',
            'pricetype',
            'bank',
            'banks',
        ]);

        return response()->json([
            'success' => true,
            'data' => $sales,
        ], 200);
    }

    public function indexPublic(Request $request): JsonResponse
    {
        $user = Auth()->user()->load('customers');


        $customersIds = $user->customers->pluck('id');

        $sales = Sale::query()->whereIn('customer_id', $customersIds)->get();

        return response()->json([
            'success' => true,
            'data' => $sales,
        ], 200);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
        $rules = [
            'total' => 'required|numeric',
            'sale_type' => 'required|string',
            'payment_type' => 'required|string',
            'customer_id' => 'nullable|exists:customers,id',
            'user_id' => 'nullable|exists:users,id',
            'pricetype_id' => 'nullable|exists:pricetypes,id',
            'bank_id' => 'nullable|exists:banks,id',
            'sale_category' => 'nullable|string:purchase,quotation,layaway',
            'shipping_total' => 'required|numeric',
            'coupon_id' => 'nullable|exists:coupons,id',
            // 'bank_id' => 'nullable|integer',
            'sale_details.*' => 'required|array',
            // 'sale_details.*.product_id' => 'required|exists:products,id',
            'sale_details.*.quantity' => 'required|integer',
            'sale_details.*.price' => 'required|numeric',
            'sale_details.*.product' => 'required|array',
            'sale_details.*.product.id' => 'required|exists:products,id',
            'sale_details.*.stock_id' => 'nullable|exists:stocks,id',
            'sale_details.*.size_id' => 'nullable|exists:sizes,id',



            'banks.*' => 'nullable|array',
            'banks.*.bank_id' => 'required|exists:banks,id',
            'banks.*.amount' => 'required|numeric',

        ];

        $validator = \Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();
        \DB::beginTransaction();
        try {
            $is_full_payment = isset($data['banks']) ? array_sum(array_column($data['banks'], 'amount')) == $data['total'] : false;

            $is_full_payment = $data['payment_type'] == 'Pago en efectivo' ? true : $is_full_payment;

            $is_full_payment = isset($data['sale_category']) && $data['sale_category'] != 'purchase' ? false : $is_full_payment;

            $discount = 0;

            $sale_total = array_sum(array_column($data['sale_details'], 'price'));

            if (isset($data['coupon_id'])) {
                $coupon_query = Coupon::query();

                $coupon_query->where('id', $data['coupon_id']);
                $coupon_query->where('status', 'active');

                $coupon = $coupon_query->first();

                \Log::info($coupon);

                if (!$coupon) {
                    throw new \Exception("Cupón no válido o ya alcanzó su límite de uso");
                }

                $coupon->update([
                    'used' => $coupon->used + 1,
                ]);

                if ($coupon->used == $coupon->use_limit) {
                    $coupon->update([
                        'status' => 'inactive',
                    ]);
                }

                if ($coupon->expires_at < now()) {
                    throw new \Exception("El cupón ya expiró");
                }

                if ($coupon->used > $coupon->use_limit) {
                    throw new \Exception("El cupón ya alcanzó su límite de uso");
                }

                if ($coupon->type == 'percent') {
                    $discount = $sale_total * ($coupon->value / 100);
                } else {
                    $discount = $coupon->value;
                }
            }

            $sale = Sale::create([
                'total' => $sale_total - $discount,
                'sale_type' => $data['sale_type'],
                'customer_id' => isset($data['customer_id']) ? $data['customer_id'] : null,
                'user_id' => isset($data['user_id']) ? $data['user_id'] : null,
                'pricetype_id' => isset($data['pricetype_id']) ? $data['pricetype_id'] : null,
                'payment_type' => isset($data['payment_type']) ? $data['payment_type'] : null,
                'bank_id' => isset($data['bank_id']) ? $data['bank_id'] : null,
                'sale_category' => isset($data['sale_category']) ? $data['sale_category'] : null,
                'shipping_total' => $data['shipping_total'],
                'coupon_id' => isset($data['coupon_id']) ? $data['coupon_id'] : null,
                'status' => $is_full_payment ? 'paid' : 'pending',
            ]);

            if (isset($data['sale_details'])) {
                foreach ($data['sale_details'] as $sale_details) {
                    $sale_detail = $sale->sale_details()->create([
                        'product_id' => $sale_details['product']['id'],
                        // $sale_details['product_id'],
                        'quantity' => $sale_details['quantity'],
                        'total' => $sale_details['price'],
                        'cart_item' => json_encode($sale_details['product']),
                        'stock_id' => $sale_details['stock_id'],
                        'size_id' => $sale_details['size_id'],

                    ]);

                    // if is quotation not update stock
                    if ($sale->sale_category == 'quotation') {
                        continue;
                    }

                    $sale_detail->stock->update([
                        'stock' => $sale_detail->stock->stock - $sale_detail->quantity,
                    ]);

                    // if sale stock has 0 stock, throw error
                    if ($sale_detail->stock->stock < 0) {
                        throw new \Exception("No hay stock suficiente para el producto {$sale_detail->product->name} en color {$sale_detail->stock->color->name}");
                    }
                }
            }

            if (isset($data['banks']) && $sale->sale_category != 'quotation') {
                foreach ($data['banks'] as $bank) {
                    $sale->banks()->attach($bank['bank_id'], ['amount' => $bank['amount']]);
                }
            }



            \DB::commit();

            $sale->load('sale_details', 'coupon');

            return response()->json([
                'success' => true,
                'data' => $sale,
            ], 201);

        } catch (\Throwable $th) {
            //throw $th;
            \DB::rollBack();
            throw $th;
        }

    }

    public function storePublic(Request $request)
    {
        //
        $rules = [
            'total' => 'required|numeric',
            'sale_type' => 'required|string',
            'payment_type' => 'required|string',
            'customer' => 'required|array',
            'customer.name' => 'required|string',
            'customer.phone_number' => 'nullable|string',
            'customer.address' => 'nullable|string',
            'user_id' => 'nullable|exists:users,id',
            'pricetype_id' => 'nullable|exists:pricetypes,id',
            'bank_id' => 'nullable|exists:banks,id',
            'sale_category' => 'nullable|string:purchase,quotation,layaway',
            'shipping_total' => 'required|numeric',
            'coupon_id' => 'nullable|exists:coupons,id',
            // 'bank_id' => 'nullable|integer',
            'sale_details.*' => 'required|array',
            // 'sale_details.*.product_id' => 'required|exists:products,id',
            'sale_details.*.quantity' => 'required|integer',
            'sale_details.*.price' => 'required|numeric',
            'sale_details.*.product' => 'required|array',
            'sale_details.*.product.id' => 'required|exists:products,id',
            'sale_details.*.stock_id' => 'nullable|exists:stocks,id',
            'sale_details.*.size_id' => 'nullable|exists:sizes,id',

            'banks.*' => 'nullable|array',
            'banks.*.bank_id' => 'required|exists:banks,id',
            'banks.*.amount' => 'required|numeric',

        ];

        $validator = \Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();
        \DB::beginTransaction();
        try {

            //if User is logged in take the customer_id from the user
            if (Auth()->user()) {
                $data['user_id'] = Auth()->user()->id;
                $data['customer_id'] = Auth()->user()->customers()->where('type', 'store')->first()->id;
            } else {
                $data['customer']['type'] = 'guest';
                $customer = \App\Models\Customer::create($data['customer']);
                $data['customer_id'] = $customer->id;
            }


            $is_full_payment = isset($data['banks']) ? array_sum(array_column($data['banks'], 'amount')) == $data['total'] : false;

            $is_full_payment = $data['payment_type'] == 'Pago en efectivo' ? true : $is_full_payment;

            $is_full_payment = isset($data['sale_category']) && $data['sale_category'] != 'purchase' ? false : $is_full_payment;

            $discount = 0;

            $sale_total = array_sum(array_column($data['sale_details'], 'price'));

            if (isset($data['coupon_id'])) {
                $coupon_query = Coupon::query();

                $coupon_query->where('id', $data['coupon_id']);
                $coupon_query->where('status', 'active');

                $coupon = $coupon_query->first();

                \Log::info($coupon);

                if (!$coupon) {
                    throw new \Exception("Cupón no válido o ya alcanzó su límite de uso");
                }

                $coupon->update([
                    'used' => $coupon->used + 1,
                ]);

                if ($coupon->used == $coupon->use_limit) {
                    $coupon->update([
                        'status' => 'inactive',
                    ]);
                }

                if ($coupon->expires_at < now()) {
                    throw new \Exception("El cupón ya expiró");
                }

                if ($coupon->used > $coupon->use_limit) {
                    throw new \Exception("El cupón ya alcanzó su límite de uso");
                }

                if ($coupon->type == 'percent') {
                    $discount = $sale_total * ($coupon->value / 100);
                } else {
                    $discount = $coupon->value;
                }
            }


            $sale = Sale::create([
                'total' => $sale_total - $discount,
                'sale_type' => $data['sale_type'],
                'customer_id' => $data['customer_id'],
                'user_id' => isset($data['user_id']) ? $data['user_id'] : null,
                'pricetype_id' => isset($data['pricetype_id']) ? $data['pricetype_id'] : null,
                'payment_type' => isset($data['payment_type']) ? $data['payment_type'] : null,
                'bank_id' => isset($data['bank_id']) ? $data['bank_id'] : null,
                'sale_category' => isset($data['sale_category']) ? $data['sale_category'] : null,
                'shipping_total' => $data['shipping_total'],
                'coupon_id' => isset($data['coupon_id']) ? $data['coupon_id'] : null,
                'status' => $is_full_payment ? 'paid' : 'pending',
            ]);

            if (isset($data['sale_details'])) {
                foreach ($data['sale_details'] as $sale_details) {
                    $sale_detail = $sale->sale_details()->create([
                        'product_id' => $sale_details['product']['id'],
                        // $sale_details['product_id'],
                        'quantity' => $sale_details['quantity'],
                        'total' => $sale_details['price'],
                        'cart_item' => json_encode($sale_details['product']),
                        'stock_id' => $sale_details['stock_id'],
                        'size_id' => $sale_details['size_id'],
                        
                    ]);

                    // if is quotation not update stock
                    if ($sale->sale_category == 'quotation') {
                        continue;
                    }

                    $sale_detail->stock->update([
                        'stock' => $sale_detail->stock->stock - $sale_detail->quantity,
                    ]);

                    // if sale stock has 0 stock, throw error
                    if ($sale_detail->stock->stock < 0) {
                        throw new \Exception("No hay stock suficiente para el producto {$sale_detail->product->name} en color {$sale_detail->stock->color->name}");
                    }
                }
            }

            if (isset($data['banks']) && $sale->sale_category != 'quotation') {
                foreach ($data['banks'] as $bank) {
                    $sale->banks()->attach($bank['bank_id'], ['amount' => $bank['amount']]);
                }
            }

            \DB::commit();

            $sale->load('sale_details');

            return response()->json([
                'success' => true,
                'data' => $sale,
            ], 201);

        } catch (\Throwable $th) {
            //throw $th;
            \DB::rollBack();
            throw $th;
        }

    }

    public function addPaymentToSale(Request $request, Sale $sale)
    {
        $rules = [
            'amount' => 'required|numeric|min:0.01',
            'bank_id' => 'required|exists:banks,id',
        ];

        $validator = \Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();




        \DB::beginTransaction();
        try {
            $sale->banks()->attach($data['bank_id'], ['amount' => $data['amount']]);

            if ($sale->pending_applied_to_balance) {
                $sale->customer()->increment('balance', $data['amount']);
            }

            if ($sale->pending_total < 0) {
                throw new \Exception("El monto ingresado es mayor al monto pendiente");
            }

            $sale->load('banks');

            $sale->update([
                'status' => $sale->pending_total == 0 ? 'paid' : 'pending',
            ]);

            \DB::commit();

            return response()->json([
                'success' => true,
                'data' => $sale,
            ], 200);
        } catch (\Throwable $th) {
            \DB::rollBack();
            throw $th;
        }

    }

    /**
     * Display the specified resource.
     */
    public function show(Sale $sale)
    {
        $sale->load([
            'sale_details',
            'sale_details.product',
            'sale_details.product.pricetypes',
            'sale_details.stock.color',
            'sale_details.stock.size',
            'customer',
            'user',
            'pricetype',
            'bank',
            'banks',
        ]);
        return response()->json([
            'success' => true,
            'data' => $sale,
        ]);
    }
    public function showPublic(Sale $sale)
    {
        $sale->load([
            'sale_details',
            'sale_details.product',
            'sale_details.product.pricetypes',
            'sale_details.stock.color',
            'sale_details.stock.size',
            'customer',
            'user',
            'pricetype',
            'bank',
            'banks',
        ]);
        return response()->json([
            'success' => true,
            'data' => $sale,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Sale $sale)
    {
        // $rules = [
        //     'total' => 'required|numeric',
        //     'sale_type' => 'required|string',
        //     'payment_type' => 'nullable|string',
        //     'bank_id' => 'nullable|exists:banks,id',
        //     'customer_id' => 'nullable|exists:customers,id',
        //     'user_id' => 'nullable|exists:users,id',
        //     'pricetype_id' => 'nullable|exists:pricetypes,id',
        //     // 'sale_details.*' => 'string',
        // ];

        $rules = [
            'total' => 'required|numeric',
            'sale_type' => 'required|string',
            'payment_type' => 'required|string',
            'customer_id' => 'nullable|exists:customers,id',
            'user_id' => 'nullable|exists:users,id',
            'pricetype_id' => 'nullable|exists:pricetypes,id',
            'sale_category' => 'nullable|string:purchase,quotation,layaway',
            'shipping_total' => 'required|numeric',
            'status' => 'nullable|string:paid,pending,cancelled'
        ];


        $validator = \Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();

        $sale->update([
            'total' => $data['total'],
            'sale_type' => $data['sale_type'],
            'customer_id' => isset($data['customer_id']) ? $data['customer_id'] : null,
            'user_id' => isset($data['user_id']) ? $data['user_id'] : null,
            'pricetype_id' => isset($data['pricetype_id']) ? $data['pricetype_id'] : null,
            'payment_type' => isset($data['payment_type']) ? $data['payment_type'] : null,
            'sale_category' => isset($data['sale_category']) ? $data['sale_category'] : null,
            'shipping_total' => $data['shipping_total'],
            'status' => $data['status']
        ]);

        if (isset($data['new_sale_details'])) {
            foreach ($data['new_sale_details'] as $sale_details) {
                $sale->sale_details()->create([
                    'product_id' => $sale_details['product_id'],
                    'quantity' => $sale_details['quantity'],
                    'stock_id' => $sale_details['stock_id'],
                    'size_id' => $sale_details['size_id'],
                ]);
            }
        }

        $sale->load('sale_details');

        return response()->json([
            'success' => true,
            'data' => $sale,
        ], 200);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Sale $sale)
    {
        $deleted = $sale->delete();
        $responseCode = $deleted ? 200 : 404;
        $message = $deleted ? 'Sale deleted' : 'Sale not found';
        $success = $deleted ? true : false;

        return response()->json(
            [
                'success' => $success,
                "message" => $message,
                'data' => $sale,
            ],
            $responseCode
        );
    }
}
