<?php

namespace App\Http\Controllers\Sales;

use App\Http\Controllers\Controller;
use App\Models\Sale_detail;
use App\Models\Sale;
use Illuminate\Http\Request;

class SaleDetailController extends Controller
{
    public function addSaleDetail(Request $request, Sale $sale)
    {
        $rules = [
            'quantity' => 'required|integer',
            'price' => 'required|numeric',
            'product' => 'required|array',
            'product.id' => 'required|exists:products,id',
            'stock_id' => 'nullable|exists:stocks,id',
            'size_id' => 'nullable|exists:sizes,id',

        ];
        $validator = \Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();

        \DB::beginTransaction();
        try {
            $sale_detail = $sale->sale_details()->create([
                'quantity' => $data['quantity'],
                'total' => $data['price'],
                'product_id' => $data['product']['id'],
                'cart_item' => json_encode($data['product']),
                'stock_id' => $data['stock_id'],
                'size_id' => $data['size_id'],
            ]);

            if ($sale->sale_category != 'quotation') {

                $sale_detail->stock->update([
                    'stock' => $sale_detail->stock->stock - $sale_detail->quantity,
                ]);

                // if sale stock has 0 stock, throw error
                if ($sale_detail->stock->stock < 0) {
                    throw new \Exception("No hay stock suficiente para el producto {$sale_detail->product->name} en color {$sale_detail->stock->color->name}");
                }
            }


            \DB::commit();

            $sale->load('sale_details');

            return response()->json([
                'success' => true,
                'data' => $sale,
            ], 200);
        } catch (\Throwable $th) {
            \DB::rollBack();
            throw $th;
        }
    }

    public function updateSaleDetail(Request $request, Sale $sale, Sale_detail $sale_detail)
    {

        if ($sale->id != $sale_detail->sale_id) {
            return response()->json([
                'success' => false,
                'sale_id' => $sale->id,
                'sale_detail_id' => $sale_detail->sale_id,
                'message' => 'Validation error',
                'errors' => 'Sale id and sale detail id does not match'
            ], 400);
        }

        $rules = [
            'quantity' => 'required|integer',
            'price' => 'required|numeric',
            'product' => 'required|array',
            'product.id' => 'required|exists:products,id',
            'stock_id' => 'nullable|exists:stocks,id',
            'size_id' => 'nullable|exists:sizes,id',
        ];
        $validator = \Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();

        \DB::beginTransaction();
        try {
            $is_return = $sale_detail->quantity > $data['quantity'];
            $diff = abs($data['quantity'] - $sale_detail->quantity);

            $sale_detail->update([
                'quantity' => $data['quantity'],
                'total' => $data['price'],
                'cart_item' => json_encode($data['product']),
                'product_id' => $data['product']['id'],
                'stock_id' => $data['stock_id'],
                'size_id' => $data['size_id'],
            ]);


            if ($is_return && $sale->sale_category != 'quotation') {
                $sale_detail->stock->update([
                    'stock' => $sale_detail->stock->stock + $diff,
                ]);
            }

            if (!$is_return && $sale->sale_category != 'quotation') {
                $sale_detail->stock->update([
                    'stock' => $sale_detail->stock->stock - $diff,
                ]);
            }
            // if sale stock has 0 stock, throw error
            if ($sale_detail->stock->stock < 0) {
                throw new \Exception("No hay stock suficiente para el producto {$sale_detail->product->name} en color {$sale_detail->stock->color->name}");
            }

            \DB::commit();

            $sale->load('sale_details');

            return response()->json([
                'success' => true,
                'is_return' => $is_return,
                'quantity' => $data['quantity'],
                'sale_detail_quantity' => $sale_detail->quantity,
                'diff' => $diff,
                'data' => $sale,
            ], 200);
        } catch (\Throwable $th) {
            \DB::rollBack();
            throw $th;
        }
    }

    public function destroySaleDetail(Request $request, Sale $sale, Sale_detail $sale_detail)
    {
        if ($sale->id != $sale_detail->sale_id) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => 'Sale id and sale detail id does not match'
            ], 400);
        }

        \DB::beginTransaction();
        try {
            if ($sale->sale_category != 'quotation') {
                $sale_detail->stock->update([
                    'stock' => $sale_detail->stock->stock + $sale_detail->quantity,
                ]);
            }

            $sale_detail->delete();

            \DB::commit();

            return response()->json([
                'success' => true,
                'data' => $sale_detail,
            ], 200);
        } catch (\Throwable $th) {
            \DB::rollBack();
            throw $th;
        }
    }
}
