<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Contracts\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): JsonResponse
    {
        $users = User::all();

        return response()->json([
            'success' => true,
            'data' => $users
        ], 200);
    }
    public function indexPublic(): JsonResponse
    {
        $users = User::query()
            ->where(['type' => 'seller', 'show_in_shop' => true])
            ->get();

        return response()->json([
            'success' => true,
            'data' => $users
        ], 200);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        $rules = [
            'name' => 'required|string',
            'email' => 'required|email|unique:users,email',
            'pin' => 'required|string|min:4|unique:users,pin',
            'password' => 'required|string',
            'phone_number' => 'nullable|string',
            'type' => 'required|in:admin,seller,customer',
            'show_in_shop' => 'nullable|boolean',
        ];

        $validator = \Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 400);
        }

        $data = $request->all();

        $data['password'] = bcrypt($request->password);

        $user = User::create($data);

        return response()->json([
            'success' => true,
            'data' => $user
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user): JsonResponse
    {
        return response()->json([
            'success' => true,
            'data' => $user
        ], 200);
    }

    public function commissionsBySeller(Request $request)
    {


        $rules = [
            'from' => 'required|date',
            'to' => 'required|date',
            'user_id' => 'nullable|exists:users,id',
        ];


        $validator = \Validator::make($request->input(), $rules);
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $user_condition = [];
        if (request('user_id')) {
            $user_condition = ['id' => request('user_id')];
        }

        $sellers = User::with([
            'sales.sale_details' => (function (Builder $query) {
                $query->whereBetween('created_at', [request('from'), request('to')]);
            })
        ])
            ->where($user_condition)
            ->get();

        $sellers_with_commission = array_map(
            function ($seller) {
                $product_count = array_sum(
                    array_map(
                        function ($sale) {
                            $product_count = array_sum(
                                array_map(
                                    function ($sale_detail) {
                                        return $sale_detail['quantity'];
                                    },
                                    $sale['sale_details']
                                )
                            );
                            return $product_count;
                        },
                        $seller['sales']
                    )
                );
                $commission = $product_count * 1 - 500;

                return [
                    'user_id' => $seller['id'],
                    'user_name' => $seller['name'],
                    'product_count' => $product_count,
                    'commission_amount' => $commission > 0 ? $commission : 0,

                ];
            },
            $sellers->toArray()
        );

        return response()->json([
            'success' => true,
            // 'sellersCount' => count($sellers),
            'sellers' => $sellers,
            'data' => $sellers_with_commission,
        ], 200);


    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user): JsonResponse
    {
        $rules = [
            'name' => 'string',
            'email' => 'email|unique:users,email',
            'pin' => 'string|min:4|unique:users,pin,' . $user->id,
            'password' => 'nullable|string',
            'phone_number' => 'string',
            'type' => 'required|in:admin,seller,customer',
            'show_in_shop' => 'nullable|boolean',
        ];

        $validator = \Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 400);
        }

        $data = $validator->validated();

        if (isset($data['password'])) {
            $data['password'] = bcrypt($request->password);
        }

        $user->update($data);

        return response()->json([
            'success' => true,
            'data' => $user
        ], 200);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user): JsonResponse
    {

        $deleted = $user->delete();
        $responseCode = $deleted ? 200 : 404;
        $message = $deleted ? 'User deleted' : 'User not found';
        $success = $deleted ? true : false;

        return response()->json(
            [
                'success' => $success,
                "message" => $message,
                'data' => $user,
            ],
            $responseCode
        );

    }
}