<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Product extends Model
{
    use HasFactory;
    protected $fillable = [
        'name',
        'slug',
        'category_id',
        'subcategory_id',
        'description',
        'supplier_name',
        'purchase_price',
        'is_top_product',
        'show_in_shop',
    ];

    protected $casts = [
        'is_top_product' => 'boolean',
        'show_in_shop' => 'boolean',
    ];

    protected $appends = [
        'is_favorite',
    ];
    protected static function boot()
    {
        parent::boot();
        static::creating(function ($product) {
            $product->slug = $product->generateSlug();
        });
    }


    public function category()
    {
        return $this->belongsTo(Category::class);
    }
    public function subcategory()
    {
        return $this->belongsTo(Subcategory::class);
    }
    public function stocks()
    {
        return $this->hasMany(Stock::class);
    }

    public function pricetypes()
    {
        return $this->belongsToMany(Pricetype::class)->withTimestamps()->withPivot('price');
    }

    public function files()
    {
        return $this->belongsToMany(File::class)->withTimestamps()->withPivot('stock_id');
    }

    private function generateSlug()
    {
        $slug = \Str::slug($this->name);
        $count = Product::whereRaw("slug RLIKE '^{$slug}(-[0-9]+)?$'")->count();
        return $count ? "{$slug}-{$count}" : $slug;
    }

    public function customersMarkedAsFavorite()
    {
        return $this->belongsToMany(Customer::class, 'favorite_products');
    }

    public function getIsFavoriteAttribute()
    {
        $user = \Auth::user();

        if (!$user) {
            return false;
        }

        $customer = $user->customers->first();

        return $this->customersMarkedAsFavorite->contains($customer);
    }


}
